/* public/js/positions-index.js */
(() => {
  const CSRF = document.querySelector('meta[name="csrf-token"]')?.content || '';

  // ===== Helpers
  const $ = (sel, el = document) => el.querySelector(sel);
  const $$ = (sel, el = document) => Array.from(el.querySelectorAll(sel));

  function showLoading() { $('#loadingOverlay')?.style.setProperty('display', 'flex'); }
  function hideLoading() { $('#loadingOverlay')?.style.removeProperty('display'); }

  // ===== Sidebar mobile open (tombol di header)
  $('#openSidebarMobile')?.addEventListener('click', () => {
    // tombol yang sudah ada di layout
    document.getElementById('mobile-menu-button')?.click();
  });

  // ===== Filter autosubmit (debounce utk search)
  (function initFilters(){
    const form = $('#filterForm');
    if (!form) return;

    const search = $('.search-input', form);
    const selects = $$('.filter-select', form);

    let t;
    if (search) {
      search.addEventListener('input', () => {
        clearTimeout(t);
        t = setTimeout(() => form.submit(), 500);
      });
    }
    selects.forEach(s => s.addEventListener('change', () => form.submit()));
  })();

  // ===== Dropdown (event delegation)
  document.addEventListener('click', (e) => {
    // open/close dropdown
    const btn = e.target.closest('.action-btn');
    if (btn) {
      e.preventDefault();
      const menu = btn.nextElementSibling;
      $$('.dropdown-menu').forEach(m => { if (m !== menu) m.classList.remove('show'); });
      menu?.classList.toggle('show');
      return;
    }

    // click outside → close all
    if (!e.target.closest('.action-dropdown')) {
      $$('.dropdown-menu').forEach(m => m.classList.remove('show'));
    }
  });

  // ===== Delete (event delegation)
  document.addEventListener('click', async (e) => {
    const btn = e.target.closest('.btn-delete-position');
    if (!btn) return;

    e.preventDefault();
    const id     = btn.dataset.positionId;
    const name   = btn.dataset.positionName || 'posisi';
    const total  = Number(btn.dataset.totalApps || 0);
    const active = Number(btn.dataset.activeApps || 0);
    const url    = btn.dataset.deleteUrl || `/positions/${id}`;

    const html = `
      Anda akan menghapus <b>${escapeHtml(name)}</b>.<br>
      Total aplikasi: <b>${total}</b> (aktif: <b>${active}</b>).
    `;

    const res = await Swal.fire({
      title: 'Hapus posisi?',
      html,
      icon: 'warning',
      showCancelButton: true,
      confirmButtonColor: '#dc2626',
      cancelButtonColor: '#6b7280',
      confirmButtonText: 'Hapus',
      cancelButtonText: 'Batal',
      reverseButtons: true,
      focusCancel: true
    });

    if (!res.isConfirmed) return;

    try {
      btn.disabled = true;
      showLoading();

      // Gunakan POST + _method DELETE (Laravel-friendly)
      const r = await fetch(url, {
        method: 'POST',
        headers: {
          'X-CSRF-TOKEN': CSRF,
          'Accept': 'application/json',
          'Content-Type': 'application/json',
          'X-Requested-With': 'XMLHttpRequest'
        },
        body: JSON.stringify({ _method: 'DELETE' })
      });

      const data = await r.json().catch(() => ({}));
      hideLoading();

      if (!r.ok || data.success === false) {
        throw new Error(data.message || `HTTP ${r.status}`);
      }

      await Swal.fire({ title: 'Berhasil', text: data.message || 'Posisi dihapus.', icon: 'success' });
      location.reload();
    } catch (err) {
      hideLoading();
      console.error(err);
      Swal.fire({ title: 'Gagal', text: 'Tidak dapat menghapus posisi.', icon: 'error' });
    } finally {
      btn.disabled = false;
    }
  });

  // ===== Toggle status (event delegation)
  document.addEventListener('click', async (e) => {
    const btn = e.target.closest('.btn-toggle-status');
    if (!btn) return;

    e.preventDefault();
    const id     = btn.dataset.positionId;
    const name   = btn.dataset.positionName || 'posisi';
    const action = btn.dataset.action; // "open" | "close"
    const total  = Number(btn.dataset.totalApps || 0);
    const active = Number(btn.dataset.activeApps || 0);

    let title, html, confirmText, confirmColor, icon;

    if (action === 'open') {
      title = 'Aktifkan Posisi?';
      icon = 'question';
      confirmText = 'Ya, Aktifkan';
      confirmColor = '#10b981';
      html = `Posisi <b>${escapeHtml(name)}</b> akan dibuka dan menerima aplikasi baru.`;
    } else {
      title = 'Nonaktifkan Posisi?';
      icon = 'warning';
      confirmText = 'Ya, Nonaktifkan';
      confirmColor = '#f59e0b';
      html = `
        Posisi <b>${escapeHtml(name)}</b> akan ditutup untuk aplikasi baru.<br>
        Total aplikasi: <b>${total}</b> (aktif: <b>${active}</b>).
      `;
    }

    const res = await Swal.fire({
      title, html, icon,
      showCancelButton: true,
      confirmButtonText: confirmText,
      cancelButtonText: 'Batal',
      confirmButtonColor: confirmColor,
      cancelButtonColor: '#6b7280',
      reverseButtons: true
    });
    if (!res.isConfirmed) return;

    try {
      btn.disabled = true;
      showLoading();

      const r = await fetch(`/positions/${id}/toggle-status`, {
        method: 'POST',
        headers: {
          'X-CSRF-TOKEN': CSRF,
          'Accept': 'application/json',
          'Content-Type': 'application/json',
          'X-Requested-With': 'XMLHttpRequest'
        },
        body: JSON.stringify({ action })
      });
      const data = await r.json().catch(() => ({}));
      hideLoading();

      if (!r.ok || data.success === false) {
        throw new Error(data.message || `HTTP ${r.status}`);
      }

      await Swal.fire({ title: 'Berhasil', text: data.message || 'Status diperbarui.', icon: 'success' });
      location.reload();
    } catch (err) {
      hideLoading();
      console.error(err);
      Swal.fire({ title: 'Gagal', text: 'Tidak dapat mengubah status.', icon: 'error' });
    } finally {
      btn.disabled = false;
    }
  });

  // ===== Utils
  function escapeHtml(s) {
    return String(s).replace(/[&<>"']/g, (m) => ({
      '&': '&amp;', '<': '&lt;', '>': '&gt;', '"': '&quot;', "'": '&#39;'
    }[m]));
  }
})();
